@* The Insert Procedure.
We treat now the insertion procedure as described in
Section~\ref{the insertion process}.
For the insertion of a point $x$, we distinguish three cases:
\begin{itemize}
\item $x$ is the first point to be inserted.
\item $x$ is a dimension jump (and not the first point to be inserted).
\item $x$ is not a dimension jump.
\end{itemize}

@<Member f...@>+=
void Triangulation::insert(const d_rat_point &x)
{
  dic_item dic_x;
// add |x| to the points already inserted and store its position in |item_x|
  if (x.dim()!=dmax)
    error_handler(99,"chull: incorrect dimension");

  list_item item_x = coordinates.append(x);
  order_nr[item_x]=co_nr++; 
  //save the insertion order of |x|
  // store position of |x| in |coordinates| in a |dictionary|
  if ((dic_x = co_index.lookup(x)) == nil) @|
   { @+ dic_x = co_index.insert(x, new list<list_item>); @+}

  co_index.inf(dic_x)->append(item_x);


  if (dcur==-1) { // |x| is the first point to be inserted
    @<Initialize the triangulation@>@;
  }
  else if ((dcur < dmax) && is_dimension_jump(x)) { // see Section \ref{is_dimension_jump}
	@<Dimension jump@>@;
  }
  else {
	@<Non-dimension jump@>@;
  }
}

@ When the first point $x$ is inserted, we must initialize our
triangulation, that means, we must build the first simplices by hand.
This is easy to do. When we only have one point, the simplicial
complex consists of two simplices: the origin simplex, containing $x$
as peak, and an outer simplex |outer_simplex| having the anti-origin
as its peak.  They both point to one another in a natural way. The
origin simplex has no base facet by definition, and because |dcur| is
|0| |outer_simplex| has a $(-1)$-dimensional base facet, that means, it
has no base facet either. The center point of the origin simplex is
clearly |x|. Also the |simplex[item_x]| value is clear.

@<Initialize th...@>=
  Simplex *outer_simplex; // a pointer to the outer simplex

  dcur =0; // we jump from dimension -1 to dimension 0
  origin_simplex=new Simplex(dmax);
  origin_simplex->this_item=all_simplices.append(origin_simplex);
  outer_simplex=new Simplex(dmax);
  outer_simplex->this_item=all_simplices.append(outer_simplex);



  origin_simplex->vertices[0]=item_x; // |x| is the only point and the peak
  origin_simplex->neighbors[0]=outer_simplex;
  origin_simplex->opposite_vertices[0]=0;

  outer_simplex->vertices[0]=anti_origin;
  outer_simplex->neighbors[0]=origin_simplex;
  outer_simplex->opposite_vertices[0]=0;

  quasi_center=x;

  simplex[item_x]=origin_simplex;


@ We discuss now how to handle insertions that are not dimension
jumps.
If the current dimension is zero, we append |x| to the points
interior to the |origin_simplex|, since it is the same point
as the only one we already inserted. Otherwise	
we first compute the set of all $x$-visible hull facets.
This is described in detail in Section~\ref{search_methods}.
As a result of this step, we get in |visible_simplices| the list
of unbounded simplices whose base facets see $x$, or in |inner_simplex|
the simplex in which $x$ in located if $x$ lies in the interior of
the current hull.
If |visible_simplices| is empty, then $x$ lies within the current hull and we
have to add it to the simplex it lies in (needed for later deletion).
Otherwise, we have to modify some simplices and to add some
new ones as described in Section \ref{the insertion process}.
Also the
neighborhood information has to be updated.
@<Non-dimension jump@>=

  // when we come here in dim 0, we inserted the same point twice
  if (dcur > 0) {
    find_visible_facets(x); // see Section \ref{find_visible_facets}
    if (! visible_simplices.empty()) {
      list<Simplex *> NewSimplices;
	// Simplices created to store horizon ridges
      Simplex *S;
      forall(S, visible_simplices)
      { /* For each $x$-visible facet $F$ of $CH(\pi_{i-1})$ alter
	   the simplex $S(F\cup\{\overline{O}\})$ of
	   $\overline{\Delta}(\pi_{i-1})$ into $S(F\cup\{x_i\})$.
	   Note that $\overline{O}$ is the peak, i.e., |S->vertices[0]|. */
	S->vertices[0] = item_x;
	simplex[item_x]=S;
        @<For each horizon ridge add the new simplex@>@;
      }
      visible_simplices.clear();
      @<Update the neighborhood relationship@>@;
    }
   else  // we have to add $x$ to the simplex it lies in
    {
     position[item_x] = inner_simplex->points.append(item_x);
     simplex[item_x]=inner_simplex;
     inner_simplex=nil;
    }
  }
  else {
    position[item_x] = origin_simplex->points.append(item_x);
    simplex[item_x]=origin_simplex;
  }


@
We now describe how to update the neighborhood relationship and
to compute the equations of the base facets of the new simplices.

\begin{zitat}{\cite{CMS}, p.\ 8}
At this point, we have
found the current hull facets seeing $x$,
in the form of the simplices whose base facets see $x$
and with the anti-origin as their peak vertex. Let ${\cal V}$
be the set of such simplices.
Now we update $T$ by altering these simplices and
creating some others.  The alteration is simply to
replace the anti-origin with $x$ in every simplex in ${\cal V}$.

The new simplices correspond to new hull facets.
Such facets are the hull of $x$ and
a horizon ridge $f$; a {\em horizon ridge} is a $(d-2)$--dimensional
face of \cor\  with the property that exactly one of the two incident
hull facets sees $x$. Each horizon ridge $f$ gives rise to a new
simplex $A_f$ with base facet $\conv(f \cup \{x\})$ and peak $\Obar$.
For each horizon ridge of \cor\  there is a non-base facet $G$ of a
simplex in ${\cal V}$ such that $x$ does not see the base facet of
the other simplex incident to the facet $G$. Thus the set of horizon
ridges is easily determined.
\end{zitat}

The figures \ref{before} and \ref{after} illustrate the situation. In
figure \ref{before}, $x$ sees the facets conv$(f,g)$ and conv $(g,h)$.
There are two horizon ridges: the points $f$ and $h$. The non-base
facet $G$ of the above text is the segment $s$ which $x$ does not see.
In figure
\ref{after}, $x$ has been inserted. Two new unbounded simplices
corresponding to the two horizon ridges have been added.
\begin{figure}[htbp]
\begin{center}
\input{before.latex}
\end{center}
\caption{\label{before}Before $x$ is inserted}
\end{figure}

\begin{figure}[htbp]
\begin{center}
\input{after.latex}
\end{center}
\caption{\label{after}After $x$ has been inserted}
\end{figure}


We find all horizon ridges incident to an updated simplex $S$ with
$x$-visible base facet by testing all its neighbors (except for the one
opposite to its peak) whether their base facet is $x$-visible.  If
the base facet of a neighbor is not $x$-visible, we have found a
horizon ridge $f$ and have to create a new simplex $T$ with base
facet $\conv(f \cup \{x\})$ and peak $\overline{O}$.  We collect all new
simplices in the list |NewSimplices|.

We use the index |k| to run through the neighbors of $S$.
When we have identified a horizon ridge, the vertices of the new
simplex $T$ are the vertices of $S$ with the $k$-th vertex replaced by
$x$. The peak of $T$ is the anti-origin $\overline{O}$. We could
therefore initialize the vertex set of the new simplex $T$ by

|T->vertices=S->vertices;|\\
\hspace*{\parindent}|T->vertices[k]=item_x;|\\
\hspace*{\parindent}|T->vertices[0]=anti_origin|;

\noindent
In order to facilitate the update of the neighborhood relation, we
proceed slightly differently: we make $x$ the highest numbered vertex
of $T$, i.e., we replace the second line by

|T->vertices[k]=S->vertices[dcur]|;\\
\hspace*{\parindent}|T->vertices[dcur]=item_x|;

\noindent
What are the neighbors of the new simplex $T$? The neighbor opposite to
$\overline{O}$ is $S$ and the neighbor opposite to $x$ is the neighbor
of the old $S$ (i.e., $S$ before the replacement of its peak
$\overline{O}$ by $x$) incident to $f \cup {\overline{O}}$. The
neighbors opposite to the $j$-th vertex of $T$, with $1\leq j <dcur$,
are computed in the next section.

@<For each horizon ridge add the new simplex@>=
for (int k = 1; k <= dcur; k++) {

  if (sees_facet(S->neighbors[k],0,x) <= 0) {
    // |x| doesn't see the base facet of the neighbor
    Simplex *T=new Simplex(dmax);
    T->this_item=all_simplices.append(T);
    NewSimplices.append(T);
    /* Take the vertices of $S$ as the vertices of the new simplex,
       replacing the current vertex by
       the |dcur|-th, the first by $x$ and the peak by $\overline{O}$ */
int ii;
    for (ii = 0; ii <= dcur; ii++)
      T->vertices[ii] = S->vertices[ii];
    T->vertices[k] = S->vertices[dcur];
    T->vertices[dcur] = item_x;
    T->vertices[0] = anti_origin;
    /* set the pointers to the two neighbors we already know and
       update the corresponding entries in the
       |opposite_vertices|-arrays */
    T->neighbors[dcur] = S->neighbors[k];
    T->opposite_vertices[dcur] = S->opposite_vertices[k];
    T->neighbors[0] = S;
    T->opposite_vertices[0] = k;
    /* Also set the reverse pointers from those two neighbors to the new simplex */
    S->neighbors[k]->neighbors[S->opposite_vertices[k]] = T;
    S->neighbors[k]->opposite_vertices[S->opposite_vertices[k]] = dcur;
    S->neighbors[k] = T;
    S->opposite_vertices[k] = 0;
  }
}


@ We now complete the update of the neighborhood relation.
How the neighborhood relationship has to be updated is described in
\cite{CMS} as follows.
\newcommand{\cS}{{\cal S}}

\begin{zitat}{\cite{CMS}, p.\ 8}
It remains to update the neighbor relationship. Let
$A_f= S(\conv(f \cup \{x\}),\Obar)$ be a new simplex
corresponding to horizon ridge $f$. In the old triangulation
(before adding $x$) there were two simplices $\overline{V}$ and $N$ incident
to the facet $\conv(f \cup \{\Obar\})$; $\overline{V} \in {\cal V}$
\footnote{${\cal V}$ is the set of outer simplices which see $x$} and
$N \not\in {\cal V}$. In the updated triangulation $\overline{V}$
is replaced by a new simplex $V$ that has the same base but peak $x$.
The neighbor of $A_f$ opposite to $x$ is $N$ and the neighbor
opposite to $\Obar$ is $V$.
Now consider
any vertex $q \in f$ and let $\cS = \cS_{f,q}$ be the set of
simplices with peak $x$ and including
$\vertex(f) \setminus \{q\} \cup \{x\}$ in their vertex set;
for a face $f$ we use $\vertex(f)$ to denote the set of vertices
contained in $f$. We will show that the neighbor of $A_f$ opposite
to $q$ can be determined by a simple walk through $\cS$. This walk
amounts to a rotation about the $(d-2)$--face
$\conv(\vertex(f) \setminus \{q\} \cup \{x\})$.
Note first that $V \in \cS$. Consider next any
simplex $S = S(F,x) \in \cS$. Then
$F = \conv(f \setminus \{q\} \cup \{y_1,y_2\})$ for some
vertices $y_1$ and $y_2$. Thus $S$ has at most two neighbors in
$\cS$, namely the neighbors opposite to $y_1$ and $y_2$ respectively.
Also, $V$ has at most one neighbor in $\cS$, namely the neighbor
opposite to $q$ (Note that the neighbor opposite to $y$, where
$\conv(f \cup \{y\})$ is the base facet of $V$, is the simplex
$A_f \not\in \cS$.). The neighbor relation thus induces a path
on the set $\cS$ with $V$ being one end of the path. Let $V'$
with base facet $\conv(f \setminus \{q\} \cup \{y_1,y_2\})$ be
the other end of the path. Assume that the neighbor of $V'$
opposite to $y_1$, call it $B$, does not belong to $\cS$ and
that $y_1 = q$ if $V = V'$, i.e., the path has length zero.
The simplex $B$ includes $\vertex(f) \setminus \{q\} \cup \{y_2,x\}$
in its vertex set and does not have peak $x$. Thus $B$ has peak
$\Obar$ and hence $B$ is the neighbor of $A_f$ opposite to $q$.
This completes the description of the update step.
\end{zitat}

\begin{figure}[htbp]
\begin{center}
\input{update.latex}
\end{center}
\caption{\label{updating}Updating the neighborhood relation}
\end{figure}

Figure \ref{updating} illustrates the situation described above in
the two dimensional case. $y_1'$ and $y_2'$ are the new values of
$y_1$ and $y_2$ after one rotation around $x$. This is the only
rotation to be made. Then the neighbor of $q$ with respect to $A_f$ is
found. It is $B$.

We implement the update of the neighborhood information
as follows.
For all new simplices corresponding to horizon ridges, the pointers
to the neighbors opposite to $x$ and $\overline{O}$ are already set
(cf.\ the previous section).
It remains to do the following for every new simplex $A_f$ corresponding
to horizon ridge $f$:
\begin{quote}
For all vertices~$q$ of $A_f$ except $x$ and $\overline{O}$ find
the neighbor of $A_f$ opposite to $q$ and set the corresponding
neighbor pointer.
\end{quote}
Note that we do not need to set the pointer from the neighbor
we have found to $A_f$, since the neighbor is also a new simplex
and hence this pointer will be (or has been) set anyhow.

Determining the neighbor of $A_f$ opposite to $q$ is done as follows.
We walk through the simplices $T$ along the path through $\cS$ starting
at |T = V = ||Af->neighbors[0]| as described in \cite{CMS}.
As long as $T \in \cS$
(i.e., the peak of $T$ is $x$)
we go to the neighbor $T'$ of $T$ opposite to $y_1$
(for $T = V$ we have $y_1 = q$).
The new $y_1$ is the node of $T'$ equal to the vertex $y_2$ of
$T$.
We store the indices of the vertices corresponding to $y_1$ and $y_2$
in two variables |y1| and |y2| respectively.
In $V$, $y_2$ is the vertex opposite to $\overline{O}$ with respect
to $A_f$.
If $T' \notin \cS$ (i.e., the peak of $T'$ is not $x$)
we have found the neighbor $B$ of $A_f$ opposite to $q$.

@<Update the neighborhood relationship@>=
Simplex *Af;
forall(Af, NewSimplices) {
  for (int k = 1; k < dcur; k++) {
       // for all vertices $q$ of |Af| except $x$ and $\overline{O}$ find the opposite neighbor
    Simplex *T = Af->neighbors[0];
    int y1;
    for (y1 = 0; T->vertices[y1] != Af->vertices[k] ; y1++);
	// exercise: show that we can also start with |y1=1| 
    int y2 = Af->opposite_vertices[0];
    while (T->vertices[0] == item_x) { // while $T \in \cS$
      /* find new $y_1$ */
      int new_y1;
      for (new_y1 = 0;
	   T->neighbors[y1]->vertices[new_y1] != T->vertices[y2];
	   new_y1++);
	// exercise: show that we can also start with |new_y1=1| 
      y2 = T->opposite_vertices[y1];
      T = T->neighbors[y1];
      y1 = new_y1;
    }
    Af->neighbors[k] = T; // update the neighborhood relationship
    Af->opposite_vertices[k]=y1; // update the opposite neighbor
  }
}


