
/*
meltdown -- mess up the screen
Copyright 1990 David Lemke and Network Computing Devices

Permission to use, copy, modify, distribute, and sell this software and its
documentation for any purpose is hereby granted without fee, provided that
the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation, and that the name of Network Computing Devices not be 
used in advertising or publicity pertaining to distribution of the 
software without specific, written prior permission.  Network Computing 
Devices makes no representations about the suitability of this software 
for any purpose.  It is provided "as is" without express or implied 
warranty.

NETWORK COMPUTING DEVICES DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS 
SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
IN NO EVENT SHALL NETWORK COMPUTING DEVICES BE LIABLE FOR ANY SPECIAL,
INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM 
LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE 
OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE 
OR PERFORMANCE OF THIS SOFTWARE.

Author:  	
  Dave Lemke
  lemke@ncd.com

  Network Computing Devices, Inc
  350 North Bernardo Ave
  Mountain View, CA 94043

meltdown.c 1.2 90/02/22
meltdown.c 1.3 90/03/31 defines for VMS
meltdown.c 1.4 94/03/26 - djhjr - port to DVX, cursor & event handling,
                                  code consolidation
meltdown.c 1.5 94/04/17 - djhjr - screen saver option
meltdown.c 1.6 94/06/28 - djhjr - port to QNX, changed TYPEOF(heights)
                                  from short to int
*/

#include <X11/Xlib.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifndef NUL
#define NUL				'\x00'
#endif

#if defined(VMS) || defined(__QNX__)
#define random			rand
#define srandom			srand
#endif

#define	MIN_SIZE		10
#define	MAX_SIZE		100

#define	MIN_DIST		10

#define	MIN_WIDTH		30
#define	WIDTH_ADD		20

#define	FINISHED		50

#define KBD_EVENTS		KeyPressMask
#define PTR_EVENTS		ButtonPressMask | PointerMotionMask
#define ALL_EVENTS		KBD_EVENTS | PTR_EVENTS

#define	rnd(x)			(random() % (x))

#ifndef min
#define	min(a, b)		(((a) < (b)) ? (a) : (b))
#endif
#ifndef max
#define	max(a, b)		(((a) > (b)) ? (a) : (b))
#endif

#define NOCRSR_WIDTH	32
#define NOCRSR_HEIGHT	32

int meltdown(Bool use_planes);
void usage(void);
int errout(Display *display, XErrorEvent *error);

Display *dpy;
Window win;
GC copygc, fillgc;
XEvent report;
int	screen;
char *prgnam;

static char nocrsr_bits[] =
	{
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
	};

void main(int argc, char **argv)
{
	XSetWindowAttributes xswat;
	XGCValues gcvals;
	XColor blkclr, xctclr;
	Colormap clrmap;
	Pixmap nocrsr_src;
	Cursor nocrsr;
	Bool use_planes = False;
	Bool scrn_saver = False;
	unsigned long vmask;
	int i;
	char *a, *display = NULL;

	prgnam = argv[0];
	if ((a = strrchr(argv[0], '/')))
		prgnam = a+1;
#ifdef MSDOS
	if ((a = strrchr(argv[0], '\\')))
		prgnam = a+1;
	if ((a = strrchr(argv[0], '.')))
		*a = NUL;
#endif

	for (i = 1; i < argc; i++)
	{
		if (strncmp(argv[i], "-d", 2) == 0 && argv[i+1])
			display = argv[++i];
		else
			if (strncmp(argv[i], "-p", 2) == 0)
				use_planes = True;
			else
				if (strncmp(argv[i], "-s", 2) == 0)
					scrn_saver = True;
				else
					usage();
	}

	XSetErrorHandler(errout);

	if (!(dpy = XOpenDisplay(display)))
	{
		fprintf(stderr, "%s:  XOpenDisplay(%s) error\n", prgnam, display);
		exit(2);
	}

	screen = DefaultScreen(dpy);
	clrmap = DefaultColormap(dpy, screen);

	xswat.override_redirect = True;
	xswat.event_mask = ALL_EVENTS;
	xswat.do_not_propagate_mask = ALL_EVENTS;
	vmask = CWOverrideRedirect | CWDontPropagate;
	win = XCreateWindow(dpy, RootWindow(dpy, screen), 0, 0, 
	  DisplayWidth(dpy, screen), DisplayHeight(dpy, screen),
	  0, CopyFromParent, CopyFromParent, CopyFromParent, vmask,
	  &xswat);
	XMapWindow(dpy, win);

	gcvals.graphics_exposures = False;
	gcvals.foreground = 1;
	gcvals.background = 0;
	copygc = XCreateGC(dpy, win, 
	  GCForeground | GCBackground | GCGraphicsExposures, &gcvals);
	gcvals.foreground = BlackPixel(dpy, screen);
	fillgc = XCreateGC(dpy, win, GCForeground, &gcvals);

	if (!XAllocNamedColor(dpy, clrmap, "black", &blkclr, &xctclr))
	{
		fprintf(stderr, "%s:  XAllocNamedColor(\"black\") error\n", prgnam);
		exit(3);
	}
    nocrsr_src = XCreateBitmapFromData(dpy, win,
	  nocrsr_bits, NOCRSR_WIDTH, NOCRSR_HEIGHT);
    nocrsr = XCreatePixmapCursor(dpy, nocrsr_src, nocrsr_src,
	  &blkclr, &blkclr, 0, 0);
	XDefineCursor(dpy, win, nocrsr);

	XGrabKeyboard(dpy, win, False, GrabModeAsync, GrabModeAsync, CurrentTime);
	XGrabPointer(dpy, win, False, PTR_EVENTS, GrabModeAsync, GrabModeAsync,
	  None, None, CurrentTime);

	if (meltdown(use_planes) == 1)
		return;

	if (scrn_saver == True)
		XForceScreenSaver(dpy, ScreenSaverActive);

	XMaskEvent(dpy, ALL_EVENTS, &report);

	return;
}

int meltdown(Bool use_planes)
{
	int	i, xloc, yloc, dist, size, width, depth = 1, finished = 0, retval = 0;
	int DpyPlanes, DpyWidth, DpyHeight;
	int **heights;

	DpyPlanes = DisplayPlanes(dpy, screen);
	DpyWidth = DisplayWidth(dpy, screen);
	DpyHeight = DisplayHeight(dpy, screen);

	heights = (int **) malloc(DpyPlanes);

	if (use_planes == True)
		depth = DisplayPlanes(dpy, screen);

	for (i = 0; i < depth; i++)
		heights[i] = (int *) calloc(sizeof(int),DpyWidth);

	depth = 0;

	srandom(getpid());

	while (!retval)
	{
		if (use_planes == True)
			depth = rnd(DpyPlanes);

		width = rnd(MIN_WIDTH) + WIDTH_ADD;

/* give values near edges a better chance */
		xloc = rnd(DpyWidth + MIN_WIDTH) - MIN_WIDTH;
		if ((xloc + width) > DpyWidth)
			xloc = DpyWidth - width;
		else
			if (xloc < 0)
				xloc = 0;

		yloc = DpyHeight;
		for (i = xloc; i < (xloc + width); i++)
			yloc = min(yloc, heights[depth][i]);
		if (yloc == DpyHeight)
			continue;

		dist = rnd(yloc/10 + MIN_DIST);
		size = rnd(max(yloc + MIN_SIZE, MAX_SIZE));

		if (use_planes == True)
		{
			XSetPlaneMask(dpy, copygc, 1 << depth);
			XSetPlaneMask(dpy, fillgc, 1 << depth);
		}

		XCopyArea(dpy, win, win, copygc, xloc, yloc, width, size,
		  xloc, yloc + dist);
		XFillRectangle(dpy, win, fillgc, xloc, yloc, width, dist);

		yloc += dist;

		for (i = xloc; i < (xloc + width); i++)
		{
		    if ((heights[depth][i] < (DpyHeight - MIN_SIZE))
			  && (yloc >= (DpyHeight - MIN_SIZE)))
				finished++;

		    heights[depth][i] = max(heights[depth][i], yloc);
		}

		if (XCheckMaskEvent(dpy, ALL_EVENTS, &report) == True)
			retval = 2;

		if (finished >= (DpyWidth - FINISHED))
			retval = 1;
	}

	for (i = 0; i < depth; i++)
		free(heights[i]);
	free(heights);

	return (retval - 1);
}

void usage(void)
{
	fprintf(stderr, "Usage:  %s [-display <display>] [-planes] [-saver]\n",
	  prgnam);
	exit(1);
}

int errout(Display *display, XErrorEvent *error)
{
	char msg[80];

	XGetErrorText(display, error->error_code, msg, 80);
	fprintf(stderr, "%s:  %s error\n", prgnam, msg);
	exit(4);
}


